#!/usr/bin/env python3
"""
Test script to verify the carbonate equilibrium calculator works correctly
"""

import eh22tools_312 as eh

def test_carbeq():
    """Test the carbeq function with example values"""
    print("Testing carbonate equilibrium calculator...")
    print("=" * 60)

    # Example surface seawater conditions
    S = 35.0      # Salinity (PSS-78)
    T = 15.0      # Temperature (°C)
    P = 0         # Pressure (dbar)
    DIC = 2100    # Dissolved Inorganic Carbon (µmol kg⁻¹)
    Alk = 2300    # Total Alkalinity (µeq kg⁻¹)

    print("\nInput Parameters:")
    print(f"  Salinity:    {S} PSS-78")
    print(f"  Temperature: {T} °C")
    print(f"  Pressure:    {P} dbar")
    print(f"  DIC:         {DIC} µmol kg⁻¹")
    print(f"  Alkalinity:  {Alk} µeq kg⁻¹")

    # Calculate carbonate equilibrium
    fCO2, pH, CO2, HCO3, CO3, omega_cal, omega_arag = eh.carbeq(S, T, P, DIC, Alk)

    print("\nResults:")
    print(f"  fCO₂:          {fCO2:.2f} µatm")
    print(f"  pH:            {pH:.4f} (total scale)")
    print(f"  CO₂:           {CO2:.2f} µmol kg⁻¹")
    print(f"  HCO₃⁻:         {HCO3:.2f} µmol kg⁻¹")
    print(f"  CO₃²⁻:         {CO3:.2f} µmol kg⁻¹")
    print(f"  Ω Calcite:     {omega_cal:.3f}")
    print(f"  Ω Aragonite:   {omega_arag:.3f}")

    print("\nInterpretation:")
    if omega_cal < 1:
        print(f"  ⚠️  Undersaturated with respect to calcite (Ω = {omega_cal:.3f} < 1)")
    else:
        print(f"  ✓  Supersaturated with respect to calcite (Ω = {omega_cal:.3f} > 1)")

    if omega_arag < 1:
        print(f"  ⚠️  Undersaturated with respect to aragonite (Ω = {omega_arag:.3f} < 1)")
    else:
        print(f"  ✓  Supersaturated with respect to aragonite (Ω = {omega_arag:.3f} > 1)")

    print("\n" + "=" * 60)
    print("✓ Test completed successfully!")

    return True

if __name__ == '__main__':
    try:
        test_carbeq()
    except Exception as e:
        print(f"\n❌ Error: {e}")
        import traceback
        traceback.print_exc()
