#!/usr/bin/env python3
"""
Flask web application for Carbonate Equilibrium Calculator
"""

from flask import Flask, request, jsonify, send_from_directory
import eh22tools_312 as eh
import numpy as np

app = Flask(__name__)

@app.route('/')
def index():
    """Serve the main HTML page"""
    return send_from_directory('.', 'index.html')

@app.route('/eh22tools_312.py')
def download_python_module():
    """Serve the Python module for download"""
    return send_from_directory('.', 'eh22tools_312.py', as_attachment=True)

@app.route('/calculate', methods=['POST'])
def calculate():
    """
    Calculate carbonate equilibrium from input parameters

    Expected JSON input:
    {
        "S": float,    # Salinity (PSS-78)
        "T": float,    # Temperature (°C, ITS-90)
        "P": float,    # Pressure (dbar)
        "DIC": float,  # Dissolved Inorganic Carbon (µmol kg⁻¹)
        "Alk": float   # Total Alkalinity (µeq kg⁻¹)
    }

    Returns JSON with calculated values:
    {
        "fCO2": float,       # Fugacity of CO2 (µatm)
        "pH": float,         # pH (total pH scale)
        "CO2": float,        # CO2 concentration (µmol kg⁻¹)
        "HCO3": float,       # HCO3 concentration (µmol kg⁻¹)
        "CO3": float,        # CO3 concentration (µmol kg⁻¹)
        "omega_cal": float,  # Calcite saturation state
        "omega_arag": float  # Aragonite saturation state
    }
    """
    try:
        # Get input data from JSON request
        data = request.get_json()

        # Validate that all required fields are present
        required_fields = ['S', 'T', 'P', 'DIC', 'Alk']
        for field in required_fields:
            if field not in data:
                return jsonify({'error': f'Missing required field: {field}'}), 400

        # Extract values
        S = float(data['S'])
        T = float(data['T'])
        P = float(data['P'])
        DIC = float(data['DIC'])
        Alk = float(data['Alk'])

        # Validate ranges (basic sanity checks)
        if S < 0 or S > 50:
            return jsonify({'error': 'Salinity should be between 0 and 50 PSS-78'}), 400
        if T < -5 or T > 50:
            return jsonify({'error': 'Temperature should be between -5 and 50 °C'}), 400
        if P < 0:
            return jsonify({'error': 'Pressure should be >= 0 dbar'}), 400
        if DIC < 0:
            return jsonify({'error': 'DIC should be >= 0 µmol kg⁻¹'}), 400
        if Alk < 0:
            return jsonify({'error': 'Alkalinity should be >= 0 µeq kg⁻¹'}), 400

        # Call the carbeq function
        fCO2, pH, CO2, HCO3, CO3, omega_cal, omega_arag = eh.carbeq(S, T, P, DIC, Alk)

        # Convert numpy types to Python native types for JSON serialization
        result = {
            'fCO2': float(fCO2),
            'pH': float(pH),
            'CO2': float(CO2),
            'HCO3': float(HCO3),
            'CO3': float(CO3),
            'omega_cal': float(omega_cal),
            'omega_arag': float(omega_arag)
        }

        return jsonify(result), 200

    except ValueError as e:
        return jsonify({'error': f'Invalid input value: {str(e)}'}), 400
    except Exception as e:
        return jsonify({'error': f'Calculation error: {str(e)}'}), 500

@app.route('/health')
def health():
    """Health check endpoint"""
    return jsonify({'status': 'ok'}), 200

if __name__ == '__main__':
    print("=" * 60)
    print("Carbonate Equilibrium Calculator - Web Interface")
    print("=" * 60)
    print("\nStarting Flask server...")
    print("Open your browser and navigate to: http://localhost:5000")
    print("\nPress Ctrl+C to stop the server")
    print("=" * 60)
    app.run(debug=True, host='0.0.0.0', port=5000)
